import subprocess
import os

class SyncManager:
    """Manages pacman repository sync and system updates"""

    def __init__(self):
        self.pacman_bin = "/usr/bin/pacman"
        self._verify_pacman()

    def _verify_pacman(self):
        """Verify pacman is available"""
        if not os.path.exists(self.pacman_bin):
            raise Exception("Pacman not found. Is this an Arch-based system?")

    def sync_repositories(self):
        """
        Sync package databases (pacman -Syy)
        Forces refresh of all package databases
        """
        try:
            result = subprocess.run(
                [self.pacman_bin, "-Syy", "--noconfirm"],
                capture_output=True,
                text=True,
                check=True
            )
            return result.stdout

        except subprocess.CalledProcessError as e:
            error_msg = e.stderr if e.stderr else str(e)
            raise Exception(f"Failed to sync repositories:\n{error_msg}")
        except FileNotFoundError:
            raise Exception("Pacman executable not found")
        except Exception as e:
            raise Exception(f"Unexpected error during sync: {str(e)}")

    def update_system(self):
        """
        Update system packages (pacman -Syu)
        Syncs databases and upgrades packages
        """
        try:
            result = subprocess.run(
                [self.pacman_bin, "-Syu", "--noconfirm"],
                capture_output=True,
                text=True,
                check=True
            )
            return result.stdout

        except subprocess.CalledProcessError as e:
            error_msg = e.stderr if e.stderr else str(e)
            raise Exception(f"Failed to update system:\n{error_msg}")
        except FileNotFoundError:
            raise Exception("Pacman executable not found")
        except Exception as e:
            raise Exception(f"Unexpected error during update: {str(e)}")

    def check_updates(self):
        """
        Check for available updates without installing them
        Returns list of packages that can be updated
        """
        try:
            result = subprocess.run(
                [self.pacman_bin, "-Qu"],
                capture_output=True,
                text=True,
                check=False  # Non-zero exit is normal when no updates
            )

            if result.returncode == 0:
                # Parse output to get list of updateable packages
                updates = []
                for line in result.stdout.strip().split('\n'):
                    if line:
                        updates.append(line)
                return updates
            else:
                return []

        except Exception as e:
            raise Exception(f"Failed to check for updates: {str(e)}")

    def clean_cache(self):
        """
        Clean package cache (pacman -Sc)
        Removes old package files from cache
        """
        try:
            result = subprocess.run(
                [self.pacman_bin, "-Sc", "--noconfirm"],
                capture_output=True,
                text=True,
                check=True
            )
            return result.stdout

        except subprocess.CalledProcessError as e:
            error_msg = e.stderr if e.stderr else str(e)
            raise Exception(f"Failed to clean cache:\n{error_msg}")
        except Exception as e:
            raise Exception(f"Unexpected error during cache clean: {str(e)}")
